/* ***********************************************************************
   *                                                                     *
   * trigio.c                                                            *
   * Christian Zurhorst, BID R&D, 30 Jul. 1998                           *
   *                                                                     *
   * trigger IO sequencer functions                                      *
   *                                                                     *
   ***********************************************************************

*/

#include <typedefs.h>

#include <dynamic.h>
#include <errcapi.h>
#include <iocommon.h>
#include <perfseq.h>
#include <regconst.h>
#include <regx32.h>
#include <regx33.h>
#include <session.h>
#include <trigio.h>


/*****************************************************************************/
/* This call initialises the descriptor table of the sequencer (trigger IO)  */
/* of the selected measure                                                   */
/*****************************************************************************/

b_errtype EXPORT BestTrigIOSeqPropDefaultSet(b_handletype handle)
{
  b_errtype	err;
  
  /* first check if handle is good     */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  /* send command to performance option hardware                             */
  if (BestIs2925(handle))
  {
    err = B_E_NOT_E2925A;
  }
  else
  {
    /* New 2x protocol */
    err = BestBasicCommand(handle, CMD_TRIGIO_PROPDEFAULTSET,
      NULL, IN_TRIGIO_PROPDEFAULTSET, NULL, NULL);
  }

  B_ERRETURN(err);

}


/*****************************************************************************/
/* This call sets a generic trigger IO property (integer based)              */
/*                                                                           */
/*****************************************************************************/

b_errtype EXPORT BestTrigIOGenPropSet(b_handletype handle,
    b_trigioseqgenproptype trigiogenprop,
    b_int32 value)
{
  b_errtype	err;
  b_int8 zw[IN_TRIGIO_GENPROPSET];
  b_int8ptr bp;
  b_int32 zwl;
  /* property value                       */

  /* first check if handle is good and check license                    */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (BestIs2925(handle))
  {
    err = B_E_NOT_E2925A;
  }
  else
  {
    /* New 2x protocol */
    zwl = (b_int32) trigiogenprop;
    bp = BestLong2Stream(zw, &zwl, 1UL);  /* then take the property */
    (void) BestLong2Stream(bp, &value, 1UL);  /* and as last the value  */
    err = BestBasicCommand(handle, CMD_TRIGIO_GENPROPSET, zw,
      IN_TRIGIO_GENPROPSET, NULL, NULL);
  }

  B_ERRETURN(err);
}
/*****************************************************************************/
/* This call gets a generic trigger IO property (integer based)              */
/*                                                                           */
/*****************************************************************************/

b_errtype EXPORT BestTrigIOGenPropGet(b_handletype handle,
    b_trigioseqgenproptype trigiogenprop,
    b_int32 * value)
{
  /* property value                       */
  b_errtype	err;
  b_int8 zw[IN_TRIGIO_GENPROPGET];
  b_int32 zwl;
  b_int16 outsize;
  b_int8 outbuf[OUT_TRIGIO_GENPROPGET];
  /* first check if handle is good and check license                    */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (BestIs2925(handle))
  {
    err = B_E_NOT_E2925A;
  }
  else
  {
    /* New 2x protocol */
    zwl = (b_int32) trigiogenprop;
    (void) BestLong2Stream(zw, &zwl, 1UL);  /* copy the property in the
                                             * stream */
    outsize = OUT_TRIGIO_GENPROPGET;
    err = BestBasicCommand(handle, CMD_TRIGIO_GENPROPGET,
      zw, IN_TRIGIO_GENPROPGET, outbuf, &outsize);
    (void) BestStream2Long(value, outbuf, 1UL);
  }

  B_ERRETURN(err);
}
/*****************************************************************************/
/* This call sets all generic properties of the selected measure to default  */
/*****************************************************************************/

b_errtype EXPORT BestTrigIOGenPropDefaultSet(b_handletype handle)
{
  b_errtype	err;
  int i;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  /* send command to performance option hardware                             */
  if (BestIs2925(handle))
  {
    err = B_E_NOT_E2925A;
  }
  else
  {
    for (i = 0; i < B_TRIGIOSEQGENPROP_SIZE; i++)
    {
      err = BestTrigIOGenPropSet(handle, (b_trigioseqgenproptype) i, 0UL);
      if (err != B_E_OK)
        break;
    }
  }

  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call sets an integer based transient property in the sequencer       */
/* descriptor table                                                          */
/*****************************************************************************/

b_errtype EXPORT BestTrigIOSeqTranPropSet(
    b_handletype handle,
    b_int32 transient,
    b_trigioseqtranproptype trigioseqtranprop,
    b_int32 value)
{
  b_errtype	err;
  b_int8 zw[12];
  b_int8ptr bp;
  b_int32 zwl;
  /* transient, property, property value  */

  /* first check if handle is good                                           */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  /* send command, property parameter and data to performance option hardware */

  if (BestIs2925(handle))
  {
    err = B_E_NOT_E2925A;
  }
  else
  {
    /* New 2x protocol */
    /* first copy the transient */
    bp = BestLong2Stream(zw, &transient, 1UL);  /* into the bytestream */
    zwl = (b_int32) trigioseqtranprop;
    bp = BestLong2Stream(bp, &zwl, 1UL);  /* then take the property */
    (void) BestLong2Stream(bp, &value, 1UL);  /* and as the last take tye
                                               * value */
    err = BestBasicCommand(handle, CMD_TRIGIO_TRANPROPSET,
      zw, IN_TRIGIO_TRANPROPSET, NULL, NULL);
  }

  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call initialises a transient in the sequencer descriptor table       */
/*****************************************************************************/

b_errtype EXPORT BestTrigIOSeqTranPropDefaultSet(b_handletype handle,
    b_int32 transient)
{
  b_errtype	err;
  b_int8 zw[4];
  /* first check if handle is good     */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  /* send command to performance option hardware                             */

  if (BestIs2925(handle))
  {
    err = B_E_NOT_E2925A;
  }
  else
  {
    /* New 2x protocol */
    (void) BestLong2Stream(zw, &transient, 1UL);  /* and then the transient */
    err = BestBasicCommand(handle, CMD_TRIGIO_TRANPROPDEFAULTSET,
      zw, IN_TRIGIO_TRANPROPDEFAULTSET, NULL, NULL);
  }

  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call sets a string based transient property in the sequencer         */
/* descriptor table of the trigger io sequencer                              */
/*****************************************************************************/

b_errtype EXPORT BestTrigIOSeqTranCondPropSet(
    b_handletype handle,
    b_int32 transient,
    b_trigioseqtrancondproptype trigioseqtrancondprop,
    b_charptrtype condition)
{
  B_DECLARE_FUNCNAME("BestTrigIOSeqTranCondPropSet [tiostrcondprpset]");

  b_errtype	err;
  /* char sync = '\1'; *//* start of string marker                    */
  b_int8ptr zw;                 /* the buffer for the string */
  b_int8ptr bp;
  b_int32 zwl;
  b_int32 length;
  /* first check if handle is good                                           */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);
  B_FCT_PARAM_NULL_POINTER_CHK(condition);

  if (BestIs2925(handle))
  {
    err = B_E_NOT_E2925A;
  }
  else
  {
    /* New 2x protocol */

    length = 9UL + (b_int32) strlen(condition);

    /* transient = 4 bytes, property = 4 bytes */
    /* then the string and the terminating 0 as the last byte */
    zw = (b_int8ptr) malloc((size_t) length);

    /* copy the transient */
    bp = BestLong2Stream(zw, &transient, 1UL);  /* into the bytestream */
    zwl = (b_int32) trigioseqtrancondprop;
    bp = BestLong2Stream(bp, &zwl, 1UL);  /* then take the property */

    /* and the last to copy is the condition */
    (void) BestByteCopy(bp, (b_int8ptr) condition, strlen(condition) + 1UL);

    err = BestBasicCommandVariable(handle, CMD_TRIGIO_TRANCONDPROPSET,
      zw, (b_int16) length, NULL, NULL);

    free(zw);                   /* free the allocated memory */
  }

  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call programs the  sequencer memory of the performance counter       */
/* of the selected measure                                                   */
/*****************************************************************************/

b_errtype EXPORT BestTrigIOSeqProg(b_handletype handle)
{
  b_errtype	err;
  
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  /* send command to performance option hardware                             */

  if (BestIs2925(handle))
  {
    err = B_E_NOT_E2925A;
  }
  else
  {
    /* New 2x protocol */
    err = BestBasicCommand(handle, CMD_TRIGIO_SEQPROG,
      NULL, IN_TRIGIO_SEQPROG, NULL, NULL);
  }

  B_ERRETURN(err);
}


/*****************************************************************************/
/* The Trigger IO run call is mapped to the BestPerfRun call. The reason is  */
/* that the run block in the ichiban of the perf and trigger io is the same. */
/*****************************************************************************/
b_errtype EXPORT BestTrigIORun(b_handletype handle)
{
  B_ERRETURN(BestPerfRun(handle));
}
/*****************************************************************************/
/* The Trigger IO stop call is mapped to the BestPerfStop call. The reason   */
/* is that the run block in the ichiban of perf and trigger io is the same.  */
/*****************************************************************************/
b_errtype EXPORT BestTrigIOStop(b_handletype handle)
{
  B_ERRETURN(BestPerfStop(handle));
}

